 /**
* \file: Transport.cpp
*
* \version: 0.1
*
* \release: $Name:$
*
* Includes the implementation of the transport layer to communicate
* with a mobile device via USB, BT or WiFi.
*
* \component: Unified SPI
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>

#include "Transport.h"


LOG_IMPORT_CONTEXT(uspi_fd)


namespace adit { namespace uspi {


UsbTransport::UsbTransport(const DeviceInfo& inDevInfo)
{
    mpLibUsbContext = nullptr;
    mpLibUsbDeviceHandle = nullptr;

    if (!mpLibUsbContext) {
        libusb_init(&mpLibUsbContext);
    }

    mInfo = inDevInfo;
}

UsbTransport::~UsbTransport()
{
    if (isConnected()) {
        disconnect();
    }

    if (mpLibUsbContext) {
        libusb_exit(mpLibUsbContext);
        mpLibUsbContext = nullptr;
    }
}

DiscoveryError UsbTransport::connect()
{
    DiscoveryError res = DiscoveryError::FAILURE;
    int32_t libusbErr = LIBUSB_SUCCESS;
    bool bFound = false;
    int32_t i = 0;

    libusb_device **devs = nullptr;
    libusb_device *usb_dev = nullptr;
    ssize_t num_devs = 0;


    if (!mpLibUsbContext) {
        return DiscoveryError::FAILURE;
    }

    uint32_t vendorId = mInfo.getiDeviceInfo(DSYS_IDVENDOR);
    uint32_t productId = mInfo.getiDeviceInfo(DSYS_IDPRODUCT);
    std::string serial;
    serial.assign(mInfo.getDeviceInfo(DSYS_SERIAL));

    if (!isConnected()) {

        num_devs = libusb_get_device_list(mpLibUsbContext, &devs);
        if ((num_devs < 0) || (devs == nullptr)) {
            return DiscoveryError::FAILURE;
        }
        LOGD_DEBUG((uspi_fd, "UsbTransport::%s() number of usb devices=%zd", __func__, num_devs));

        while (((usb_dev = devs[i++]) != nullptr) && (true != bFound))
        {
            struct libusb_device_descriptor desc;

            if (0 > libusb_get_device_descriptor(usb_dev, &desc)) {
                /* start next iteration of while() */
                continue;
            }

            /* check the vendorId and productId */
            if ((vendorId == desc.idVendor)
                && (productId == desc.idProduct)) {

                /* open libusb device to check if this is the correct device */
                if (LIBUSB_SUCCESS == (libusbErr = libusb_open(usb_dev, &mpLibUsbDeviceHandle))) {
                    if (desc.iSerialNumber > 0) {
                        char retrievedSerial[64];
                        int len = libusb_get_string_descriptor_ascii(mpLibUsbDeviceHandle,
                                                                     desc.iSerialNumber,
                                                                     (unsigned char*) retrievedSerial,
                                                                     sizeof(retrievedSerial));
                        /* furthermore, compare the serial number */
                        if ((len < 0) || (0 != serial.compare(retrievedSerial))) {
                            LOGD_DEBUG((uspi_fd, "UsbTransport::%s() SerialNumber=%s does not match. Start next iteration",
                                    __func__, &retrievedSerial[0]));
                            res = DiscoveryError::FAILURE;
                        } else {
                            LOGD_DEBUG((uspi_fd, "UsbTransport::%s() Device found and open", __func__));
                            bFound = true;
                        }
                    } else {
                        res = DiscoveryError::FAILURE;
                        LOG_WARN((uspi_fd, "UsbTransport::%s() No iSerialNumber available", __func__));
                    }
                } else {
                    LOG_WARN((uspi_fd, "UsbTransport::%s() libusb_open() of device 0x%X:0x%X %s failed = %d",
                            __func__, vendorId, productId, serial.c_str(), libusbErr));
                }
            } else {
                LOGD_DEBUG((uspi_fd, "UsbTransport::%s() vendorId & productId (0x%X:0x%X) does not match to (0x%X:0x%X). Start next iteration",
                        __func__, desc.idVendor, desc.idProduct, vendorId, productId));
                res = DiscoveryError::FAILURE;
            }

            if (true != bFound) {
                /* did not found our device but libusb was opened, close it */
                disconnect();
                if (nullptr != mpLibUsbDeviceHandle) {
                    LOG_WARN((uspi_fd, "UsbTransport::%s() LibUsbDeviceHandle was not set to nullptr", __func__));
                    mpLibUsbDeviceHandle = nullptr;
                }
            }
        }/* while-loop */

        /* free and unreference all devices */
        libusb_free_device_list(devs, 1);

        if (bFound) {
            LOGD_DEBUG((uspi_fd, "UsbTransport::%s() Open device 0x%X:0x%X %s with libusb_device_handle %p",
                    __func__, vendorId, productId, serial.c_str(), mpLibUsbDeviceHandle));
            res = DiscoveryError::OK;
        } else {
            LOG_WARN((uspi_fd, "UsbTransport::%s() device 0x%X:0x%X %s not available",
                    __func__, vendorId, productId, serial.c_str()));
            res = DiscoveryError::NOT_FOUND;
        }
    } else {
        LOGD_DEBUG((uspi_fd, "UsbTransport::%s() libusb_device_handle %p already open", __func__, mpLibUsbDeviceHandle));
        res = DiscoveryError::OK;
    }

    return res;
}

DiscoveryError UsbTransport::disconnect()
{
    if (!mpLibUsbContext) {
        return DiscoveryError::FAILURE;
    }
    libusb_close(mpLibUsbDeviceHandle);
    mpLibUsbDeviceHandle = nullptr;

    return DiscoveryError::OK;
}

DiscoveryError UsbTransport::transmit(uint8_t inReqType, uint8_t inReq, uint16_t inValue, uint16_t inIndex, \
                                      uint8_t *DataBuf, uint16_t DataLength, uint16_t* outTransfered, int32_t inTimeout)
{
    DiscoveryError res = DiscoveryError::FAILURE;
    int32_t libusbErr = LIBUSB_SUCCESS;

    /* example:
     * 0x40, 0x51, 0x00, 0x00, NULL, 0
     * inReqType    : 0x40 (REQ_TYPE_USB_DEVICE_TO_HOST)
     * inReq        : 0x51 (81[dec])
     * inValue      : 0x00
     * inIndex      : 0x00
     * DataBuf      : NULL
     * DataLength   : 0
     * inTimeout    : 1000
     */
    if (nullptr != mpLibUsbDeviceHandle) {

        libusbErr = libusb_control_transfer(mpLibUsbDeviceHandle, inReqType, inReq, inValue, inIndex, DataBuf, DataLength, inTimeout);

        if (libusbErr >= LIBUSB_SUCCESS) {
            LOGD_DEBUG((uspi_fd, "UsbTransport::%s()  All data send to device res = %d",
                    __func__, libusbErr));
            /* provide number of bytes actually transferred */
            *outTransfered = libusbErr;
            res = DiscoveryError::OK;
        } else if (libusbErr == LIBUSB_ERROR_PIPE) {
            /* In our use case, it is common that a request maybe return with PIPE error. */
            LOG_INFO((uspi_fd, "UsbTransport::%s()  Device does not support control request =%d",
                    __func__, libusbErr));
            res = DiscoveryError::PIPE;
        } else if (libusbErr == LIBUSB_ERROR_NO_DEVICE) {
            /* In case send() transmit a switch-command, the device disappear and libusb return NO_DEVICE
             * which is in this use case expected, but in other cases not. */
            LOG_INFO((uspi_fd, "UsbTransport::%s()  Device has been disconnected =%d",
                    __func__, libusbErr));
            res = DiscoveryError::NO_DEVICE;
        } else if (libusbErr == LIBUSB_ERROR_TIMEOUT) {
            /* In our use case, it is common that a request maybe timeout. */
            LOG_INFO((uspi_fd, "UsbTransport::%s()  Send control request timed out =%d",
                    __func__, libusbErr));
            res = DiscoveryError::TIMEOUT;
        } else {
            LOG_ERROR((uspi_fd, "UsbTransport::%s()  Send control request failed =%d",
                    __func__, libusbErr));
            res = DiscoveryError::FAILURE;
        }
    } else {
        LOG_ERROR((uspi_fd, "UsbTransport::%s()  libusb device handle not open", __func__));
    }

    return res;
}

DiscoveryError UsbTransport::receive(uint8_t *outDataBuf, uint16_t inDataLength, int32_t inTimeout)
{
    DiscoveryError res = DiscoveryError::FAILURE;
    LOG_WARN((uspi_fd, "UsbTransport::%s()  receive is not yet implemented", __func__));

    (void)outDataBuf;
    (void)inDataLength;
    (void)inTimeout;

    return res;
}

DiscoveryError UsbTransport::release()
{
    DiscoveryError res = DiscoveryError::FAILURE;
    int32_t libusbErr = LIBUSB_SUCCESS;

    if (nullptr != mpLibUsbDeviceHandle) {

        /* Perform a USB port reset to reinitialize a device.
        * The system will attempt to restore the previous configuration
        * and alternate settings after the reset has completed.
        *
        * The device will appear to be disconnected and reconnected if:
        *  - the reset fails
        *  - the descriptors change
        *  - or the previous state cannot be restored
        * This means that the device handle is no longer valid.
        * You should close it and rediscover the device.
        *
        * Return LIBUSB_ERROR_NOT_FOUND if re-enumeration is required,
        * or if the device has been disconnected
        */
        libusbErr = libusb_reset_device(mpLibUsbDeviceHandle);
        if ((LIBUSB_ERROR_NO_DEVICE == libusbErr) || (LIBUSB_ERROR_NOT_FOUND == libusbErr)) {
            /* libusb reset done and re-enumeration is required,
             * or if the device has been disconnected */
            LOGD_DEBUG((uspi_fd, "UsbTransport::%s() libusb_reset_device() of device succeed", __func__));
            res = DiscoveryError::OK;
        } else if (LIBUSB_SUCCESS == libusbErr) {
            /* If libusb_reset_device() returns with LIBUSB_SUCCESS,
            * it is an indication that the mobile device may does not
            * disconnect and reconnect. Reset not done. Trace warning */
            LOG_INFO((uspi_fd, "UsbTransport::%s() The device might not be able to reset. libusb_reset_device()=%d.",
                  __func__, libusbErr));
            res = DiscoveryError::INCOMPLETE;
        } else {
            LOG_ERROR((uspi_fd, "UsbTransport::%s() libusb_reset_device() of device failed=%d",
                  __func__, libusbErr));
            res = DiscoveryError::FAILURE;
        }
    }  else {
        LOG_ERROR((uspi_fd, "UsbTransport::%s()  libusb device handle not open", __func__));
    }

    return res;
}


} } /* namespace adit { namespace uspi { */
